#############################################################################
#############################################################################
#
# latexUtilities.tcl (sourced on demand)
#
#############################################################################
#
# Author:  Tom Scavo <trscavo@syr.edu>, Vince Darley
#
#############################################################################
#############################################################################

# Delete all unnecessary comments from the current document:
proc deleteComments {} {
    if {![dialog::yesno "Warning!  This operation can not be undone.  \
      Continue anyway?"]} { return }

    catch {
	replaceString {}
	performSearch -s -f 1 -r 1 -i 1 -m 0 -- "^\[ \t\]*%\[^\r\n\]*\r" [minPos]
	replaceAll
    }

    catch {
	replaceString {}
	performSearch -s -f 1 -r 1 -i 1 -m 0 -- "\[ \t\]+%\[^\r\n\]*" [minPos]
	replaceAll
    }

    catch {
	replaceString {\1%}
	performSearch -s -f 1 -r 1 -i 1 -m 0 -- "(\[^\\\](\\\\)*)%\[^\r\n\]*" [minPos]
	replaceAll
    }
}

# Converts all straight quotes to their TeX equivalents.
proc convertQuotes {} {
    global searchNoisily
    message "working"
    watchCursor
    set messageString "selection"
    if {[pos::compare [set start [getPos]] == [set end [selEnd]]]} {
	set messageString "document"
	set start [minPos]
	set end [maxPos]
    }
    set text [getText $start $end]
    # Convert all left double quotes:
    set convert1 [regsub -all "\(\^\|\[\ \r\t\(\[\{\]\)\"" $text {\1``} text]
    # Convert all right double quotes:
    set convert2 [regsub -all "\(\[\^\\\\\]\)\"" $text {\1''} text]
    # Convert all left single quotes:
    set convert3 [regsub -all "\(\^\|\[\ \r\t\(\[\{\]\)\'" $text {\1`} text]
    if {$convert1 || $convert2 || $convert3} {
	replaceText $start $end $text
	message "all quotes in $messageString converted"
    } else {
	if {$searchNoisily} {beep}
	message "no quotes found in $messageString"
    }
}

# Convert all dollar signs to their LaTeX equivalents:
proc convertDollarSigns {} {
    global searchNoisily
    if {[isSelection]} {
	set messageString "selection"
    } else {
	set messageString "document"
    }
    set subs2 [convertDoubleDollarSigns]
    if {$subs2 == -1} {
	beep
	alertnote "unmatched double dollar signs in $messageString"
    } else {
	set subs1 [convertSingleDollarSigns]
	if {$subs1 == -1} {
	    beep
	    alertnote "unmatched single dollar sign in $messageString"
	} elseif {$subs1 == 0 && $subs2 == 0} {
	    if {$searchNoisily} {beep}
	    message "no dollar signs found in $messageString"
	} else {
	    message "$subs1 pairs of \$\$ and $subs2 pairs of \$\$\$\$ removed from $messageString"
	}
    }
}

# Converts all $$...$$ pairs to \[...\] and returns the number of such 
# pairs converted.  If the dollar signs are unbalanced, does nothing 
# and returns -1.
proc convertDoubleDollarSigns {} {
    watchCursor
    set messageString "selection"
    if {[pos::compare [set start [getPos]] == [set end [selEnd]]]} {
	set messageString "document"
	set start [minPos]
	set end [maxPos]
    }
    set text [getText $start $end]
    set subs [regsub -all {(^|[^\\])\$\$([^$]*)\$\$} $text {\1\\[\2\\]} text]
    if {[containsDoubleDollarSigns $text]} {return -1}
    if {$subs} {
	replaceText $start $end $text
    }
    return [expr $subs]
}

# Returns true if the argument contains non-literal double dollar
# signs, and false otherwise.
proc containsDoubleDollarSigns {text} {
    return [regexp {(^|[^\\])\$\$} $text]
}

# Converts all $...$ pairs to \(...\), maintains the cursor position, 
# and returns the number of such pairs converted.  If the dollar signs 
# are unbalanced, does nothing and returns -1.
proc convertSingleDollarSigns {} {
    watchCursor
    set subs1 0; set subs2 0; set subs3 0
    set pos [getPos]; set pos2 $pos
    if {[pos::compare [set start $pos] == [set end [selEnd]]]} {
	set isSelection 0
	set start [minPos]
	set end [maxPos]
	set text1 [getText $start $pos]
	set subs1 [regsub -all {(^|[^\\])\$([^$]*)\$} $text1 {\1\\(\2\\)} text1]
	# Is there a dollar sign left over?  If so, search backward for this
	# dollar sign and prepare to do a substitution on the text to the right
	# of this dollar sign.
	if {[containsSingleDollarSign $text1]} {
	    set searchString {[^\\]\$}
	    set searchResult [search -s -n -f 0 -m 0 -i 1 -r 1 $searchString [pos::math $pos -1]]
	    set pos2 [lindex $searchResult 0]
	    set text1 [string range $text1 0 [expr {[pos::diff $pos2 [minPos]]+ (2 * $subs1)}]]
	    set pos [pos::math $pos + 2]
	}
	set text2 [getText $pos2 $end]
	set subs2 [regsub -all {(^|[^\\])\$([^$]*)\$} $text2 {\1\\(\2\\)} text2]
	# Is there a dollar sign left over?  If so, it's unbalanced.
	if {[containsSingleDollarSign $text2]} {return -1}
	append text $text1 $text2
    } else {
	set isSelection 1
	set text [getText $start $end]
	set subs3 [regsub -all {(^|[^\\])\$([^$]*)\$} $text {\1\\(\2\\)} text]
	# Is there a dollar sign left over?  If so, it's unbalanced.
	if {[containsSingleDollarSign $text]} {return -1}
    }
    if {$subs1 || $subs2 || $subs3} {
	replaceText $start $end $text
	# If there is a selection, just put it back.  Otherwise, adjust the
	# cursor position based on the number of substitutions.
	if {$isSelection} {
	    set end [getPos]
	    select $start $end
	} else {
	    goto [pos::math $pos + [expr {2 * $subs1}]]
	}
    }
    return [expr {$subs1 + $subs2 + $subs3}]
}

# Returns true if the argument contains a non-literal dollar sign,
# and false otherwise.
proc containsSingleDollarSign {text} {
    return [regexp {(^|[^\\])\$} $text]
}

